#!/bin/sh
########################################################################
########################################################################
#                                                                      #
#  This script is used by the daemon for catering device-initiated     #
#  scan requests.                                                      #
#                                                                      #
#  This script will display save-file dialog for the user to select    #
#  where scanner output will be saved to. After a file is selected     #
#  scanimage will be invoked with the parameters given to this script  #
#                                                                      #
#  Requires the following scripts:                                     #
#  (refer to PATHS section for actual filepaths)                       #
#      save dialog script                                              #
#      confirm dialog script                                           #
#      error dialog script                                             #
#      scan image script                                               # 
#                                                                      #
#  Makes use of the following environment variables                    #
#      SCAN_SAVE_DIALOG_TITLE                                          #
#      SCAN_SAVE_DIALOG_FILE_FILTER                                    #
#      SCAN_CONFIRM_DIALOG_TITLE                                       #
#      SCAN_DEFAULT_FILENAME                                           #
#                                                                      #
#  Sets the following environment variables                            #
#      SCAN_DEVICE_WITH_EDITS                                          #
#                                                                      #
#  Parameters: scanimage parameters                                    #
########################################################################


## PATHS ##

# leave this blank if the shell or C system() is going to be used
# use full path if an exec function is to be used
script_path="./"
if [ -n "${SCAN_SCRIPT_PATH}" ]; then
	script_path="${SCAN_SCRIPT_PATH}"  
fi

# Path to save file dialog script
save_dialog_script="${script_path}dialogSaveFile.sh"

# Path to confirmation dialog script
confirm_dialog_script="${script_path}dialogConfirmation.sh"

# Path to error notification dialog script
error_dialog_script="${script_path}dialogNotification.sh ERROR"

# Path to info notification dialog script
info_dialog_script="${script_path}dialogNotification.sh SUCCESS"

# Path to info notification dialog script
scanimage_script="${script_path}scanPerformScanImage.sh"

lang=`echo $LANG | sed 's/\..*$//g'`
lang2=`echo $LANG | sed 's/_.*$//g'`
if [ -f "${script_path}/strings_${lang}.sh" ]; then
    . "${script_path}/strings_${lang}.sh"
elif [ -f "${script_path}/strings_${lang2}.sh" ]; then
    . "${script_path}/strings_${lang2}.sh"
else
    . "${script_path}/strings_en.sh"
fi


## MESSAGES / LABELS ##

# Window title of save file dialog
save_dialog_title="Save"
if [ -n "${MSG_SCAN_SAVE_DIALOG_TITLE}" ]; then
	save_dialog_title="${MSG_SCAN_SAVE_DIALOG_TITLE}"
fi

# Window title of confirmation dialog 
confirm_dialog_title="Confirmation"
if [ -n "${MSG_SCAN_CONFIRM_DIALOG_TITLE}" ]; then
	save_dialog_title="${MSG_SCAN_CONFIRM_DIALOG_TITLE}"
fi

# Confirmation question for cancel save scenario
confirm_cancel_question="Are you sure you want to cancel?"
if [ -n "${MSG_SCAN_CANCEL_QUESTION}" ]; then
	confirm_cancel_question="${MSG_SCAN_CANCEL_QUESTION}"
fi

# Error message when transfer fails
scan_transfer_error="Scan data transfer failed."
if [ -n "${MSG_SCAN_TRANSFER_ERROR}" ]; then
	scan_transfer_error="${MSG_SCAN_TRANSFER_ERROR}"
fi

# Error message when save fails
scan_save_error="Saving scan output failed. Please check permissions or select another location."
if [ -n "${MSG_SCAN_SAVE_ERROR}" ]; then
	scan_save_error="${MSG_SCAN_SAVE_ERROR}"
fi

# Success message when save succeeds
scan_save_success="Scan output successfully saved."
if [ -n "${MSG_SCAN_SAVE_SUCCESS}" ]; then
	scan_save_success="${MSG_SCAN_SAVE_SUCCESS}"
fi

# yes button label 
scan_yes_label="Yes"
if [ -n "${MSG_SCAN_YES_BUTTON_LABEL}" ]; then
	scan_yes_label="${MSG_SCAN_YES_BUTTON_LABEL}"
fi

# no button label 
scan_no_label="No"
if [ -n "${MSG_SCAN_NO_BUTTON_LABEL}" ]; then
	scan_no_label="${MSG_SCAN_NO_BUTTON_LABEL}"
fi



## MAIN ##

# File filter of save file dialog
save_dialog_filter="*.*"
if [ -n "${SCAN_SAVE_DIALOG_FILE_FILTER}" ]; then
	save_dialog_filter="${SCAN_SAVE_DIALOG_FILE_FILTER}"
fi

# Default filename for scan output
default_filename="/tmp/"`date +"%Y-%m-%d_%H:%M:%S"`
if [ -n "${SCAN_DEFAULT_FILENAME}" ]; then
	default_filename="${SCAN_DEFAULT_FILENAME}"
fi

# Temp filename for scanimage output
temp_filename="/tmp/tmp_"`date +"%Y-%m-%d_%H:%M:%S"`

# Remove problem file		
rm /tmp/scan_problem.dump > /dev/null 2>&1

# Export env variable to enable scanner backend
# to send scan start message with scan edits ENABLED
export SCAN_DEVICE_WITH_EDITS=1
export SCAN_ENABLE_DEMON=1

# Execute scan image
${scanimage_script} "$@"  > "${temp_filename}"  2> /dev/null 

temp=$?

# Export env variable to enable scanner backend
# to send scan start message with scan edits DISABLED
export SCAN_DEVICE_WITH_EDITS=0


# Check if problem file created; Problem file is
# is created by scanlib when there is scan with edits
# containing image resizing/rotation changes
if  [ -f /tmp/scan_problem.dump ]; then
	rm /tmp/scan_problem.dump > /dev/null 2>&1
	${error_dialog_script} "Operation not supported. Any image resizing and rotation done on the scanner is currently not yet supported by the driver for Linux."
	exit $temp;
fi

# If scan image image failed
if [ $temp -ne 0 ]; then
	${error_dialog_script} "${scan_transfer_error}"

	# Termiinate script
	exit $temp;
fi

flag=0


# Loop until user selects a file or until he confirms
# cancel operation 
while [ 1 ]; do
	mv "${temp_filename}" "${default_filename}"
	${info_dialog_script} "${scan_save_success} ${default_filename}"
	flag=1
	break;

	# Display save file dialog 
	file=`${save_dialog_script} "${save_dialog_title}" "${save_dialog_filter}" "${default_filename}"`

	# If user has selected a file location
	if [ $? -eq 0 ]; then
		# Transfer output from temporary to user-specified location 
		mv ${temp_filename} ${file} > /dev/null 2>&1

		# If move/transfer succeeded
		if [ $? -eq 0 ]; then
			${info_dialog_script} "${scan_save_success}"
			flag=1
			break
		fi

		${error_dialog_script} "${scan_save_error}"
	fi

	# Display confirmation dialog for cancel
	${confirm_dialog_script} "${confirm_dialog_title}" "${confirm_cancel_question}" "${scan_yes_label}" "${scan_no_label}"

	# If user confirmed save cancellation
	if [ $? -eq 0 ]; then
		break;
	fi 
done

if [ $flag -eq 0 ]; then
	rm -rf ${temp_filename} > /dev/null 2>&1

	# NOTE: the value is a special exit value that would inform the 
	# calling application(in this case, the scan daemon), that the
	# script found no errors but the scan save operation was 
	# explicitly cancelled by the user 
	exit 99999
fi

exit 0

